#!/usr/bin/env python3
# -*- coding: utf-8 -*-

'''
 * Carto_photos  - cadastre.py
 * Copyright (C) [2024] [BOTREL]
 *
 * Ce programme est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier selon les termes de la licence GNU General Public License
 * telle que publiée par la Free Software Foundation, soit la version 3
 * de la licence, soit (à votre choix) toute version ultérieure.
 *
 * Ce programme est distribué dans l'espoir qu'il sera utile,
 * mais SANS AUCUNE GARANTIE ; sans même la garantie implicite de
 * QUALITÉ MARCHANDE ou d'ADÉQUATION À UN USAGE PARTICULIER. Consultez la
 * GNU General Public License pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la GNU General Public License
 * avec ce programme. Si ce n'est pas le cas, consultez <https://www.gnu.org/licenses/>.
 '''
 
import os
import sys
import time
import geopandas as gpd
import folium
from fastkml import kml
from shapely.geometry import Point
from selenium import webdriver
from selenium.webdriver.firefox.service import Service
from dotenv import load_dotenv
from webdriver_manager.firefox import GeckoDriverManager
from selenium.webdriver.firefox.options import Options
import xml.etree.ElementTree as ET
from geopy.distance import geodesic
from PIL import Image, ImageDraw, ImageFont
import warnings

# Définir le répertoire de base (où se trouve le script)
base_dir = os.path.dirname(os.path.abspath(__file__))

# Lister tous les fichiers KML dans le répertoire source
kml_files = [f for f in os.listdir(base_dir) if f.endswith('.kml')]

# Vérifier s'il y a des fichiers KML
if not kml_files:
    print("Aucun fichier .kml trouvé dans le répertoire. Le script va s'arrêter.")
    sys.exit()  # Arrêter l'exécution du script si aucun fichier kml n'est trouvé

# Fonction pour extraire les coordonnées des points de passage à partir du fichier KML
def extraire_coordonnees_kml(chemin_fichier):
    tree = ET.parse(chemin_fichier)
    root = tree.getroot()
    waypoints = []
    for placemark in root.findall('.//{http://www.opengis.net/kml/2.2}Placemark'):
        coord_str = placemark.find('.//{http://www.opengis.net/kml/2.2}coordinates').text
        coord = tuple(map(float, coord_str.split(',')))
        waypoints.append((placemark.find('.//{http://www.opengis.net/kml/2.2}name').text, coord))
    return waypoints

# Fonction pour lire le fichier KML et extraire les waypoints
def read_kml(file_path):
    with open(file_path, 'rt', encoding='utf-8') as f:
        doc = f.read()
    
    k = kml.KML()
    k.from_string(doc)
    
    features = list(k.features())
    placemarks = list(features[0].features())
    
    # Extraire les coordonnées et noms des waypoints
    data = []
    for placemark in placemarks:
        coords = placemark.geometry.coords[0]
        name = placemark.name
        data.append({
            'geometry': Point(coords[0], coords[1]),  # coords is (lon, lat)
            'Name': name
        })
    
    gdf = gpd.GeoDataFrame(data)
    return gdf

# Fonction pour créer une carte avec folium et le cadastre.gouv.fr
def create_map(gdf, bounds, zoom_start=None):
    # Ajouter une marge aux limites (bounds) pour garantir que les points les plus au nord et au sud sont bien visibles
    margin = 0.0000  # Ajuster la marge selon les besoins
    bounds_with_margin = [
        [bounds[0][0] - margin, bounds[0][1] - margin],
        [bounds[1][0] + margin, bounds[1][1] + margin]
    ]
    
    # Créer la carte avec des tuiles du cadastre.gouv.fr et un niveau de zoom initial facultatif
    fmap = folium.Map(zoom_start=zoom_start)
    folium.TileLayer(
        tiles='https://wxs.ign.fr/cadastre/geoportail/wmts?' +
              'SERVICE=WMTS&REQUEST=GetTile&VERSION=1.0.0&LAYER=CADASTRALPARCELS.PARCELS&STYLE=normal&' +
              'TILEMATRIXSET=PM&TILEMATRIX={z}&TILEROW={y}&TILECOL={x}&FORMAT=image/png',
        attr='cadastre.gouv.fr',
        name='Cadastre',
        overlay=False,
        control=True,
        max_zoom=23
    ).add_to(fmap)

    # Ajuster les limites de la carte pour inclure tous les points
    fmap.fit_bounds(bounds_with_margin)

    # Ajouter les waypoints à la carte avec des marqueurs et des étiquettes de texte personnalisées
    for _, row in gdf.iterrows():
        folium.Marker(
            location=[row.geometry.y, row.geometry.x],
            icon=folium.CustomIcon('custom_icon.png', icon_size=(31, 31)),  # Réduire la taille de l'icône à 50%
            popup=row['Name'],  # Utiliser le nom comme popup
        ).add_to(fmap)
        # Ajouter une étiquette de texte personnalisée avec le nom en taille originale et couleur noire
        folium.map.Marker(
            [row.geometry.y, row.geometry.x],
            icon=folium.DivIcon(
                icon_size=(30, 30),
                icon_anchor=(4, 17),  # Ancre au centre horizontalement et verticalement
                html=f'<div style="font-size: 13px; color: black;">{row["Name"]}</div>'
            )
        ).add_to(fmap)
    
    return fmap

# Fonction pour enregistrer la carte en PNG
def save_map_as_png(fmap, file_name):
    # Sauvegarder la carte dans un fichier HTML temporaire
    fmap.save('temp_map.html')

    # Configuration du driver Selenium
    options = Options()
    options.add_argument('--headless')

    driver = webdriver.Firefox(service=Service(GeckoDriverManager().install()), options=options)

    # Charger la carte dans le navigateur
    driver.get('file://' + os.getcwd() + '/temp_map.html')
    time.sleep(2)  # Attendre que la carte se charge complètement

    # Définir la résolution de la capture d'écran
    screen_width = driver.execute_script("return window.innerWidth;")
    screen_height = driver.execute_script("return window.innerHeight;")
    # Ajuster la taille de la fenêtre pour inclure tout le contenu de la carte
    driver.set_window_size(screen_width, screen_height + 200)  # Augmenter la hauteur pour éviter le troncage

    # Prendre une capture d'écran
    driver.save_screenshot(file_name)
    driver.quit()
    
    return screen_width, screen_height  # Retourner les dimensions de l'écran

# Fonction pour ajouter du texte à l'image PNG
def ajouter_texte_sur_image(file_name, texte, position):
    # Ouvrir l'image existante
    image = Image.open(file_name)
    draw = ImageDraw.Draw(image)
    
    # Définir la police et la taille du texte
    try:
        font_path = os.path.join('', 'DejaVuSans.ttf')
        font = ImageFont.truetype(font_path, 16)
    except IOError:
        # En cas d'erreur, utiliser une taille approximative avec la police par défaut
        font = ImageFont.load_default()
    
    # Ajouter le texte à l'image
    draw.text(position, texte, font=font, fill="black")
    
    # Sauvegarder l'image modifiée
    image.save(file_name)

# Fonction pour trouver les limites (bounds) de la carte
def trouver_limites(waypoints):
    min_lat = min(waypoints, key=lambda wp: wp[1][1])[1][1]
    max_lat = max(waypoints, key=lambda wp: wp[1][1])[1][1]
    min_lon = min(waypoints, key=lambda wp: wp[1][0])[1][0]
    max_lon = max(waypoints, key=lambda wp: wp[1][0])[1][0]
    return [[min_lat, min_lon], [max_lat, max_lon]]

# Fonction pour déterminer le niveau de zoom en fonction des bounds
def niveau_de_zoom_from_bounds(bounds, screen_width, screen_height):
    lat_diff = bounds[1][0] - bounds[0][0]
    lon_diff = bounds[1][1] - bounds[0][1]
    max_diff = max(lat_diff, lon_diff)

    if max_diff < 0.0005:
        return 21
    elif max_diff < 0.001:
        return 20
    elif max_diff < 0.005:
        return 19
    elif max_diff < 0.01:
        return 18
    elif max_diff < 0.05:
        return 17
    elif max_diff < 0.1:
        return 16
    elif max_diff < 0.5:
        return 15
    elif max_diff < 1.0:
        return 14
    elif max_diff < 5.0:
        return 13
    elif max_diff < 10.0:
        return 12
    else:
        return 11

# Fonction pour calculer la distance entre deux points
def distance_entre_points(point1, point2):
    return geodesic(point1, point2).meters

# Désactivation des warnings
warnings.filterwarnings("ignore")

# Traiter chaque fichier KML
for kml_file in kml_files:
    print(f"Traitement de la partie cadastre")
    # Chemin vers le fichier KML
    kml_file_path = os.path.join(base_dir, kml_file)

    # Lire le fichier KML
    waypoints_gdf = read_kml(kml_file_path)

    # Extraction des coordonnées des points de passage du fichier KML
    waypoints = extraire_coordonnees_kml(kml_file_path)

    # Trouver les limites (bounds) des waypoints
    bounds = trouver_limites(waypoints)

    # Créer la carte avec les limites (bounds) pour inclure tous les points
    fmap = create_map(waypoints_gdf, bounds)

    # Enregistrer la carte en PNG et obtenir les dimensions de l'écran
    output_png_file = f'{os.path.splitext(kml_file)[0]}.png'
    screen_width, screen_height = save_map_as_png(fmap, output_png_file)

    # Déterminer le niveau de zoom basé sur les bounds et les dimensions de l'écran
    niveau_zoom = niveau_de_zoom_from_bounds(bounds, screen_width, screen_height)
    
    # Re-créer la carte avec le niveau de zoom approprié
    fmap = create_map(waypoints_gdf, bounds, zoom_start=niveau_zoom)

    # Enregistrer la carte en PNG à nouveau
    save_map_as_png(fmap, output_png_file)

    # Ajouter des informations supplémentaires aux captures d'écran
    for zoom_offset, zoom_name in zip([0], ["default"]):
        adjusted_zoom = max(1, niveau_zoom + zoom_offset)  # Assurez-vous que le zoom est au moins de 1
        fmap = create_map(waypoints_gdf, bounds, zoom_start=adjusted_zoom)
        output_zoomed_png_file = f'{os.path.splitext(kml_file)[0]}_{zoom_name}.png'
        save_map_as_png(fmap, output_zoomed_png_file)
        
        # Ajouter des informations sur la distance et le niveau de zoom
        if zoom_name == "default":
            # Coordonnées des waypoints A et B
            coord_a = next((coord for name, coord in waypoints if name == 'A'), None)
            coord_b = next((coord for name, coord in waypoints if name == 'B'), None)

            if coord_a and coord_b:
                distance_ab = distance_entre_points(coord_a, coord_b)
                distance_ab_arrondie = round(distance_ab)
                # ajouter_texte_sur_image(output_zoomed_png_file, f"Distance A-B: {distance_ab_arrondie} m", position=(45, Image.open(output_zoomed_png_file).height - 45))
        
        #ajouter_texte_sur_image(output_zoomed_png_file, f"Niveau de zoom: {adjusted_zoom}", position=(45, Image.open(output_zoomed_png_file).height - 30))
    
    print(f"Cartes sauvegardées sous cadastre.png")
      # Renommer le fichier temporaire avec les modifications
os.rename('kml2_default.png', 'cadastre.png')
os.remove('kml2.png')
#os.remove('geckodriver.log')
fichier1 = "temp_map.html"

# Vérifie si le fichier existe et le supprime
if os.path.exists(fichier1):
    os.remove(fichier1)
    # print(f"{fichier1} a été supprimé.")
else:
    print(f"{fichier1} n'existe pas.")
print("Traitement terminé.")
