document.addEventListener('DOMContentLoaded', () => {
  const plan = document.getElementById('plan');
  const blackCross1 = document.getElementById('black-cross1');
  const blackCross2 = document.getElementById('black-cross2');
  const greenCross = document.getElementById('green-cross');
  const redCross = document.getElementById('red-cross');
  const distanceInput = document.getElementById('distance-input');
  const calculatedDistanceElement = document.getElementById('calculated-distance');
  const calculatedAzimuthElement = document.getElementById('calculated-azimuth');

  let selectedElement = null;
  let offsetX = 0, offsetY = 0;

  // Fonction pour calculer la distance entre deux éléments
  function calculateDistance(x1, y1, x2, y2) {
    return Math.sqrt(Math.pow(x2 - x1, 2) + Math.pow(y2 - y1, 2));
  }

  // Fonction pour calculer l'azimut
  function calculateAzimuth(x1, y1, x2, y2) {
    const angleRad = Math.atan2(y2 - y1, x2 - x1);
    let angleDeg = angleRad * (180 / Math.PI);
    angleDeg = (angleDeg + 90 + 360) % 360; // Ajoute 90° et ajuste pour obtenir une valeur entre 0 et 360°
    return angleDeg.toFixed(2); // Renvoie un azimut avec 2 décimales
  }

  // Fonction pour mettre à jour les informations de distance et d'azimut
  function updateInfo() {
    const black1Rect = blackCross1.getBoundingClientRect();
    const black2Rect = blackCross2.getBoundingClientRect();
    const greenRect = greenCross.getBoundingClientRect();
    const redRect = redCross.getBoundingClientRect();

    // Calculer la distance et l'azimut entre la croix rouge et la croix verte
    const distance = calculateDistance(greenRect.x, greenRect.y, redRect.x, redRect.y);
    const azimuth = calculateAzimuth(greenRect.x, greenRect.y, redRect.x, redRect.y);

    // Mettre à jour les valeurs affichées
    calculatedDistanceElement.textContent = `Distance : ${distance.toFixed(2)} px`;
    calculatedAzimuthElement.textContent = `Azimut : ${azimuth}°`;

    // Si l'utilisateur a entré une distance entre les croix noires, on ajuste les valeurs affichées
    const inputDistance = parseFloat(distanceInput.value);
    if (!isNaN(inputDistance) && inputDistance > 0) {
      const scalingFactor = inputDistance / calculateDistance(black1Rect.x, black1Rect.y, black2Rect.x, black2Rect.y);
      calculatedDistanceElement.textContent = `Distance : ${(distance * scalingFactor).toFixed(2)} mètres`;
    }
  }

  // Fonction pour démarrer le déplacement
  function startDrag(e) {
    selectedElement = e.target;

    const rect = selectedElement.getBoundingClientRect();

    if (e.type === 'touchstart') {
      // Gestion du toucher
      offsetX = e.touches[0].clientX - rect.left;
      offsetY = e.touches[0].clientY - rect.top;
    } else {
      // Gestion de la souris
      offsetX = e.clientX - rect.left;
      offsetY = e.clientY - rect.top;
    }

    document.addEventListener('mousemove', drag);
    document.addEventListener('touchmove', drag);
    document.addEventListener('mouseup', endDrag);
    document.addEventListener('touchend', endDrag);
  }

  // Fonction pour déplacer l'élément
  function drag(e) {
    if (!selectedElement) return;

    let clientX, clientY;

    if (e.type === 'touchmove') {
      clientX = e.touches[0].clientX;
      clientY = e.touches[0].clientY;
    } else {
      clientX = e.clientX;
      clientY = e.clientY;
    }

    // Calculer la position par rapport au conteneur du plan
    const planRect = plan.getBoundingClientRect();

    // Assurer que les croix restent dans les limites du plan
    const newX = clientX - planRect.left - offsetX;
    const newY = clientY - planRect.top - offsetY;

    selectedElement.style.left = `${Math.max(0, Math.min(newX, planRect.width - selectedElement.offsetWidth))}px`;
    selectedElement.style.top = `${Math.max(0, Math.min(newY, planRect.height - selectedElement.offsetHeight))}px`;

    updateInfo(); // Mettre à jour la distance et l'azimut
  }

  // Fonction pour arrêter le déplacement
  function endDrag() {
    document.removeEventListener('mousemove', drag);
    document.removeEventListener('touchmove', drag);
    document.removeEventListener('mouseup', endDrag);
    document.removeEventListener('touchend', endDrag);
    selectedElement = null;
  }

  // Ajout des événements pour la souris et le tactile
  [blackCross1, blackCross2, greenCross, redCross].forEach(cross => {
    cross.addEventListener('mousedown', startDrag);
    cross.addEventListener('touchstart', startDrag);
  });

  // Mettre à jour les informations initiales
  updateInfo();
});
