#!/usr/bin/env python3
# -*- coding: utf-8 -*-

'''
 * Carto_photos  - carto1.py
 * Copyright (C) [2024] [BOTREL]
 *
 * Ce programme est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier selon les termes de la licence GNU General Public License
 * telle que publiée par la Free Software Foundation, soit la version 3
 * de la licence, soit (à votre choix) toute version ultérieure.
 *
 * Ce programme est distribué dans l'espoir qu'il sera utile,
 * mais SANS AUCUNE GARANTIE ; sans même la garantie implicite de
 * QUALITÉ MARCHANDE ou d'ADÉQUATION À UN USAGE PARTICULIER. Consultez la
 * GNU General Public License pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la GNU General Public License
 * avec ce programme. Si ce n'est pas le cas, consultez <https://www.gnu.org/licenses/>.
 '''

import sys
import os
import time
import geopandas as gpd
import folium
from fastkml import kml
from shapely.geometry import Point
from selenium import webdriver
from selenium.webdriver.firefox.service import Service
from dotenv import load_dotenv
from webdriver_manager.firefox import GeckoDriverManager
from selenium.webdriver.firefox.options import Options
import xml.etree.ElementTree as ET
from geopy.distance import geodesic
from PIL import Image, ImageDraw, ImageFont
import warnings
import threading
import tkinter as tk
from tkinter import simpledialog, messagebox
import subprocess






# Fonction pour calculer la distance entre deux points
def calculer_distance_entre_deux_points(coord1, coord2):
    return geodesic(coord1, coord2).meters

# Définir le répertoire de base (où se trouve le script)
base_dir = os.path.dirname(os.path.abspath(__file__))

# Lister tous les fichiers KML dans le répertoire source
kml_files = [f for f in os.listdir(base_dir) if f.endswith('.kml')]

# Vérifier s'il y a des fichiers KML
if not kml_files:
    print("Aucun fichier .kml trouvé dans le répertoire. Le script va s'arrêter.")
    sys.exit()

# Fonction pour extraire les coordonnées des points de passage à partir du fichier KML
def extraire_coordonnees_kml(chemin_fichier):
    tree = ET.parse(chemin_fichier)
    root = tree.getroot()
    waypoints = []
    for placemark in root.findall('.//{http://www.opengis.net/kml/2.2}Placemark'):
        coord_str = placemark.find('.//{http://www.opengis.net/kml/2.2}coordinates').text
        coord = tuple(map(float, coord_str.split(',')))
        waypoints.append((placemark.find('.//{http://www.opengis.net/kml/2.2}name').text, coord))
    return waypoints

# Fonction pour lire le fichier KML et extraire les waypoints
import geopandas as gpd

def read_kml(kml_file):
    print("📥 Lecture KML via GeoPandas/Fiona…")
    try:
        gdf = gpd.read_file(kml_file, driver="KML")
        print("✅ GDF chargé :", len(gdf), "points")
        print("📌 Colonnes →", gdf.columns)
        print(gdf.head())
        return gdf
    except Exception as e:
        print("❌ Erreur lecture KML :", e)
        return gpd.GeoDataFrame()



# Fonction pour trouver les waypoints le plus au nord et le plus au sud
def waypoints_nord_sud(waypoints):
    nord = max(waypoints, key=lambda wp: wp[1][1])
    sud = min(waypoints, key=lambda wp: wp[1][1])
    return nord, sud


# Fonction pour déterminer le niveau de zoom en fonction de la distance maximale
def niveau_de_zoom(distance_max):
    if distance_max < 50:
        return 21  # Zoom de niveau 21
    elif distance_max < 100:
        return 20  # Zoom de niveau 20
    elif distance_max < 200:
        return 19  # Zoom de niveau 19
    elif distance_max < 400:
        return 18  # Zoom de niveau 18
    elif distance_max < 800:
        return 17  # Zoom de niveau 17
    elif distance_max < 1600:
        return 16  # Zoom de niveau 16
    elif distance_max < 3200:
        return 15  # Zoom de niveau 15
    elif distance_max < 6400:
        return 14  # Zoom de niveau 14
    elif distance_max < 12800:
        return 13  # Zoom de niveau 13
    else:
        return 12  # Zoom de niveau 12 ou un autre zoom par défaut



# Fonction pour calculer la distance entre deux points
def distance_entre_points(point1, point2):
    return geodesic(point1, point2).meters



# Lire le dernier choix de l'utilisateur depuis le fichier
def lire_dernier_choix(fichier):
    if os.path.exists(fichier):
        with open(fichier, 'r') as f:
            return f.read().strip()
    return None



# Fonction pour créer une carte avec folium
def create_map(gdf, bounds, zoom_start=None, tile_type='OpenStreetMap'):
    margin = 0.0000  # Ajuster la marge selon les besoins
    bounds_with_margin = [
        [bounds[0][0] - margin, bounds[0][1] - margin],
        [bounds[1][0] + margin, bounds[1][1] + margin]
    ]
    
    fmap = folium.Map(tiles=tile_type, zoom_start=zoom_start, attr='&copy; <a href="https://www.cadastre.gouv.fr/">Cadastre</a>')
    fmap.fit_bounds(bounds_with_margin)

    for _, row in gdf.iterrows():
        folium.Marker(
            location=[row.geometry.y, row.geometry.x],
            icon=folium.CustomIcon('custom_icon.png', icon_size=(31, 31)),  # Réduire la taille de l'icône à 50%
            popup=row['Name'],
        ).add_to(fmap)
        folium.map.Marker(
            [row.geometry.y, row.geometry.x],
            icon=folium.DivIcon(
                icon_size=(30, 30),
                icon_anchor=(4, 17),
                html=f'<div style="font-size: 13px; font-weight: bold; color: black;">{row["Name"]}</div>'
            )
        ).add_to(fmap)
    
    return fmap
  # Fonction pour créer une carte avec google satellite
    
def create_map2(gdf, waypoints, zoom_start=14):
    # Calculer le centre de la carte
    center = [float(gdf.geometry.y.mean()), float(gdf.geometry.x.mean())]

    # Créer la carte avec des tuiles Google Satellite
    fmap = folium.Map(location=center, zoom_start=zoom_start, tiles=None)
    folium.TileLayer(
        tiles='http://{s}.google.com/vt/lyrs=s&x={x}&y={y}&z={z}',
        attr='Google',
        max_zoom=20,
        subdomains=['mt0', 'mt1', 'mt2', 'mt3']
    ).add_to(fmap)
    
    # Ajouter les waypoints à la carte avec des étiquettes de texte en rouge et gras
    for _, row in gdf.iterrows():
        folium.Marker(
            location=[float(row.geometry.y), float(row.geometry.x)],  # Assurez-vous que les coordonnées sont des floats
            icon=folium.CustomIcon('https://raw.githubusercontent.com/pointhi/leaflet-color-markers/master/img/marker-icon-blue.png', icon_size=(20, 30))  # Icônes redimensionnées
        ).add_to(fmap)
        folium.map.Marker(
            [float(row.geometry.y), float(row.geometry.x)],  # Assurez-vous que les coordonnées sont des floats
            icon=folium.DivIcon(html=f"""<div style="font-size: 14px; color: red; font-weight: bold;">{row['Name']}</div>""")
        ).add_to(fmap)
    
    return fmap
 
 
   
 # Fonction pour créer une carte avec folium et le cadastre.gouv.fr
def create_map3(gdf, bounds, zoom_start=None):
    # Ajouter une marge aux limites (bounds) pour garantir que les points les plus au nord et au sud sont bien visibles
    margin = 0.0000  # Ajuster la marge selon les besoins
    bounds_with_margin = [
        [bounds[0][0] - margin, bounds[0][1] - margin],
        [bounds[1][0] + margin, bounds[1][1] + margin]
    ]
    
    
    

    
    
    # Créer la carte avec des tuiles du cadastre.gouv.fr et un niveau de zoom initial facultatif
    fmap = folium.Map(zoom_start=zoom_start)
    folium.TileLayer(
        tiles='https://wxs.ign.fr/cadastre/geoportail/wmts?' +
              'SERVICE=WMTS&REQUEST=GetTile&VERSION=1.0.0&LAYER=CADASTRALPARCELS.PARCELS&STYLE=normal&' +
              'TILEMATRIXSET=PM&TILEMATRIX={z}&TILEROW={y}&TILECOL={x}&FORMAT=image/png',
        attr='cadastre.gouv.fr',
        name='Cadastre',
        overlay=False,
        control=True,
        max_zoom=23
    ).add_to(fmap)

    # Ajuster les limites de la carte pour inclure tous les points
    fmap.fit_bounds(bounds_with_margin)

    # Ajouter les waypoints à la carte avec des marqueurs et des étiquettes de texte personnalisées
    for _, row in gdf.iterrows():
        folium.Marker(
            location=[row.geometry.y, row.geometry.x],
            icon=folium.CustomIcon('custom_icon.png', icon_size=(31, 31)),  # Réduire la taille de l'icône à 50%
            popup=row['Name'],  # Utiliser le nom comme popup
        ).add_to(fmap)
        # Ajouter une étiquette de texte personnalisée avec le nom en taille originale et couleur noire
        folium.map.Marker(
            [row.geometry.y, row.geometry.x],
            icon=folium.DivIcon(
                icon_size=(30, 30),
                icon_anchor=(4, 17),  # Ancre au centre horizontalement et verticalement
                html=f'<div style="font-size: 13px; color: black;">{row["Name"]}</div>'
            )
        ).add_to(fmap)
    
    return fmap   
    

# Fonction pour enregistrer la carte en PNG
def save_map_as_png(fmap, file_name):
    fmap.save('temp_map.html')
    options = Options()
    options.add_argument('--headless')
    driver = webdriver.Firefox(service=Service(GeckoDriverManager().install()), options=options)
    driver.get('file://' + os.getcwd() + '/temp_map.html')
    
    # Obtenir la taille de la fenêtre du navigateur
    screen_width = driver.execute_script("return window.innerWidth;")
    screen_height = driver.execute_script("return window.innerHeight;")
    
    # Ajuster la taille de la fenêtre pour capturer la carte complète
    driver.set_window_size(screen_width, screen_height + 200)
    
    # Sauvegarder la capture d'écran
    driver.save_screenshot(file_name)
    driver.quit()
    
    return screen_width, screen_height

# Fonction pour ajouter du texte à l'image PNG
def ajouter_texte_sur_image(file_name, texte, position, color="black", font_bold=False):
    image = Image.open(file_name)
    draw = ImageDraw.Draw(image)
    
    try:
        font_path = os.path.join('', 'DejaVuSans-Bold.ttf' if font_bold else 'DejaVuSans.ttf')
        font = ImageFont.truetype(font_path, 16)
    except IOError:
        font = ImageFont.load_default()
    
    draw.text(position, texte, font=font, fill=color)
    image.save(file_name)


# Fonction pour ajouter du texte à l'image PNG
def ajouter_texte_sur_image2(file_name, texte, position, color="yellow", font_bold=True):
    image = Image.open(file_name)
    draw = ImageDraw.Draw(image)
    
    try:
        font_path = os.path.join('', 'DejaVuSans-Bold.ttf' if font_bold else 'DejaVuSans.ttf')
        font = ImageFont.truetype(font_path, 16)
    except IOError:
        font = ImageFont.load_default()
    
    draw.text(position, texte, font=font, fill=color)
    image.save(file_name)

# Fonction pour trouver les limites (bounds) de la carte
def trouver_limites(waypoints):
    min_lat = min(waypoints, key=lambda wp: wp[1][1])[1][1]
    max_lat = max(waypoints, key=lambda wp: wp[1][1])[1][1]
    min_lon = min(waypoints, key=lambda wp: wp[1][0])[1][0]
    max_lon = max(waypoints, key=lambda wp: wp[1][0])[1][0]
    return [[min_lat, min_lon], [max_lat, max_lon]]

# Fonction pour déterminer le niveau de zoom en fonction des bounds
def niveau_de_zoom_from_bounds(bounds, max_zoom):
    lat_diff = bounds[1][0] - bounds[0][0]
    lon_diff = bounds[1][1] - bounds[0][1]
    max_diff = max(lat_diff, lon_diff)
    
    # Calculer le niveau de zoom basé sur la différence maximale
    if max_diff < 0.001:
        zoom = 20
    elif max_diff < 0.005:
        zoom = 19
    elif max_diff < 0.01:
        zoom = 18
    elif max_diff < 0.05:
        zoom = 17
    elif max_diff < 0.1:
        zoom = 16
    elif max_diff < 0.5:
        zoom = 15
    elif max_diff < 1.0:
        zoom = 14
    elif max_diff < 5.0:
        zoom = 13
    elif max_diff < 10.0:
        zoom = 12
    else:
        zoom = 11
    
    return min(zoom, max_zoom)

def lire_distance_depuis_fichier(fichier_distance):
    """
    Lit la distance depuis le fichier 'distance.txt' et retourne la valeur.
    """
    try:
        with open(fichier_distance, 'r') as file:
            # Lire la première ligne et extraire la valeur de la distance
            ligne = file.readline().strip()
            distance_str = ligne.split()[-2]  # Extraire le nombre avant 'mètres'
            distance_m = int(distance_str)
            return distance_m
    except Exception as e:
        print(f"Erreur lors de la lecture de la distance depuis le fichier : {e}")
        return None

def ajouter_texte_sur_image(image_path, texte, position):
    """
    Ajoute du texte sur une image à une position donnée.
    """
    try:
        # Ouvrir l'image
        image = Image.open(image_path)
        draw = ImageDraw.Draw(image)
        font = ImageFont.load_default()

        # Ajouter le texte à l'image
        draw.text(position, texte, font=font, fill="black")
        image.save("terrain.png")
        print(f"Texte ajouté à l'image et enregistré sous 'terrain.png'.")
    except Exception as e:
        print(f"Erreur lors de l'ajout du texte sur l'image : {e}")

def partie_1(fichier_kml):
    print("Chargement des waypoints...")
    gdf = read_kml(fichier_kml)
    waypoints = extraire_coordonnees_kml(fichier_kml)
    bounds = trouver_limites(waypoints)
    
    # Déterminer le niveau de zoom, avec un maximum de 20
    screen_width, screen_height = 800, 600  # Valeurs par défaut initiales
    zoom_start = niveau_de_zoom_from_bounds(bounds, max_zoom=20)

    print("Création de la carte...")
    fmap = create_map(gdf, bounds, zoom_start, tile_type='OpenStreetMap')

    print("Sauvegarde de la carte en PNG...")
    file_name = 'terrain.png'
    screen_width, screen_height = save_map_as_png(fmap, file_name)

    # Chemin vers le fichier 'distance.txt' dans le répertoire supérieur
    #fichier_distance = os.path.join(os.path.dirname(os.path.abspath(__file__)),  'distance.txt')

    # Lire la distance depuis le fichier 'distance.txt'
    #distance_ab = lire_distance_depuis_fichier(fichier_distance)

    #if distance_ab is not None:
        # Ajouter la distance A-B à l'image
       # print(f"Distance entre A et B : {distance_ab} mètres")
        #ajouter_texte_sur_image(file_name, f"Distance A-B: {distance_ab} m", position=(45, Image.open(file_name).height - 45))
    #else:
        #print("La distance n'a pas pu être lue depuis le fichier.")
    
    # print(f"Carte sauvegardée sous {file_name}")
    
    # Nettoyage des fichiers temporaires
    script_directory = os.path.dirname(os.path.abspath(__file__))
    chemins_temp_files = [
        os.path.join(script_directory, 'temp_map.html'),
    ]

    for chemin_du_fichier in chemins_temp_files:
        try:
            os.remove(chemin_du_fichier)
            # print(f"Le fichier {chemin_du_fichier} a été supprimé avec succès.")
        except FileNotFoundError:
            print(f"Le fichier {chemin_du_fichier} n'existe pas.")
        except Exception as e:
            print(f"Une erreur est survenue : {e}")

    print(f"Carte sauvegardée sous {file_name}")





def ajouter_texte_sur_image2b(image_path, texte, position):
    """
    Ajoute du texte en jaune sur une image à une position donnée avec la police DejaVuSans.ttf.
    """
    try:
        # Charger la police
        font_path = 'DejaVuSans-Bold.ttf'  # Assurez-vous que le fichier DejaVuSans.ttf est dans le même répertoire que le script
        font_size = 24 # Ajustez la taille de la police si nécessaire
        font = ImageFont.truetype(font_path, font_size)

        # Ouvrir l'image
        image = Image.open(image_path)
        draw = ImageDraw.Draw(image)

        # Ajouter le texte à l'image
        draw.text(position, texte, font=font, fill="yellow")
        image.save("satellite.png")
        print(f"Texte ajouté à l'image et enregistré sous 'satellite.png'.")
    except Exception as e:
        print(f"Erreur lors de l'ajout du texte sur l'image : {e}")


# Partie 2 - Utilisation des tuiles Google Satellite

def partie_2(fichier_kml):
    print("Chargement des waypoints...")
    # Lire le fichier KML et extraire les waypoints
    waypoints_gdf = read_kml(fichier_kml)
    waypoints = extraire_coordonnees_kml(fichier_kml)
    
    # Calculer la distance maximale entre les waypoints
    distance_max = max(distance_entre_points(wp1[1], wp2[1]) for wp1 in waypoints for wp2 in waypoints if wp1 != wp2)
    
    # Trouver les waypoints le plus au nord et le plus au sud
    nord, sud = waypoints_nord_sud(waypoints)
    distance_nord_sud = distance_entre_points(nord[1], sud[1])
    
    # Ajuster le niveau de zoom
    niveau_zoom = niveau_de_zoom(distance_max)
    if distance_nord_sud > distance_max / 1.45:
        niveau_zoom -= 1

    # print(f"Niveau de zoom calculé : {niveau_zoom}")

    # Créer la carte
    fmap = create_map2(waypoints_gdf, waypoints, zoom_start=niveau_zoom)

    print("Sauvegarde de la carte en PNG...")
    file_name = 'satellite.png'
    save_map_as_png(fmap, file_name)

    # Chemin vers le fichier 'distance.txt' dans le répertoire supérieur
    fichier_distance = os.path.join(os.path.dirname(os.path.abspath(__file__)),  'distance.txt')

    # Lire la distance depuis le fichier 'distance.txt'
    distance_ab = lire_distance_depuis_fichier(fichier_distance)

    if distance_ab is not None:
        # Ajouter la distance A-B à l'image
        # print(f"Distance entre A et B : {distance_ab} mètres")
        ajouter_texte_sur_image2b(file_name, f"Distance A-B: {distance_ab} m", position=(45, Image.open(file_name).height - 45))
    else:
        print("La distance n'a pas pu être lue depuis le fichier.")

    # Nettoyage des fichiers temporaires
    script_directory = os.path.dirname(os.path.abspath(__file__))
    chemins_temp_files = [
        os.path.join(script_directory, 'temp_map.html'),
    ]

    for chemin_du_fichier in chemins_temp_files:
        try:
            os.remove(chemin_du_fichier)
            # print(f"Le fichier {chemin_du_fichier} a été supprimé avec succès.")
        except FileNotFoundError:
            print(f"Le fichier {chemin_du_fichier} n'existe pas.")
        except Exception as e:
            print(f"Une erreur est survenue : {e}")

    print(f"Carte sauvegardée sous {file_name}")







# Partie 3 - Utilisation des tuiles Cadastre de cadastre.gouv.fr

def partie_3(fichier_kml):
    print("Chargement des waypoints...")    
    # Appeler le script cadastre.py
    # Détecter le système d'exploitation
    python_cmd = 'python3' if os.name != 'nt' else 'python'
    # Exécution du script avec la commande appropriée
    try:
        subprocess.run([python_cmd, 'extras/cadastre.py', fichier_kml], check=True)
        print("Le script cadastre.py a été exécuté avec succès.")
    except subprocess.CalledProcessError as e:
        print(f"Une erreur est survenue lors de l'exécution de cadastre.py : {e}")

    

# Fonction principale
def main():
    dernier_choix = lire_dernier_choix('choix_utilisateur.txt')
    

    for fichier_kml in kml_files:
        if '1' in dernier_choix:
            partie_1(fichier_kml)
        if '2' in dernier_choix:
            partie_2(fichier_kml)
        if '3' in dernier_choix:
            partie_3(fichier_kml)

if __name__ == "__main__":
    main()
